<?php
require_once '../config/database.php';

// Check admin access
if (!isset($_SESSION['admin']) || !$_SESSION['admin']) {
    redirect('../login.php');
}

$db = getDB();
$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['email_file'])) {
    $file = $_FILES['email_file'];
    $batch_name = sanitize($_POST['batch_name']);
    
    // Validate file
    $allowed_types = ['text/csv', 'text/plain', 'application/vnd.ms-excel'];
    $max_size = 10 * 1024 * 1024; // 10MB
    
    if ($file['error'] !== UPLOAD_OK) {
        $message = '<div class="alert alert-danger">File upload error.</div>';
    } elseif (!in_array($file['type'], $allowed_types)) {
        $message = '<div class="alert alert-danger">Only CSV files are allowed.</div>';
    } elseif ($file['size'] > $max_size) {
        $message = '<div class="alert alert-danger">File size exceeds 10MB limit.</div>';
    } else {
        // Create uploads directory if not exists
        $upload_dir = '../uploads/email_lists/';
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        
        $filename = uniqid() . '_' . basename($file['name']);
        $filepath = $upload_dir . $filename;
        
        if (move_uploaded_file($file['tmp_name'], $filepath)) {
            // Start background processing
            $import_id = startEmailImport($batch_name, $filename, $filepath, $_SESSION['user_id']);
            
            $message = '<div class="alert alert-success">
                File uploaded successfully! Processing started. 
                <a href="import-status.php?id=' . $import_id . '">View Import Status</a>
            </div>';
        } else {
            $message = '<div class="alert alert-danger">Failed to save uploaded file.</div>';
        }
    }
}

// Get recent imports
$recent_imports = $db->query("
    SELECT ib.*, u.name as importer_name 
    FROM email_import_batches ib 
    LEFT JOIN users u ON ib.imported_by = u.id 
    ORDER BY ib.import_date DESC 
    LIMIT 10
");

function startEmailImport($batch_name, $filename, $filepath, $user_id) {
    $db = getDB();
    
    // Create batch record
    $stmt = $db->prepare("
        INSERT INTO email_import_batches (batch_name, filename, imported_by, status) 
        VALUES (?, ?, ?, 'processing')
    ");
    $stmt->bind_param("ssi", $batch_name, $filename, $user_id);
    $stmt->execute();
    $batch_id = $db->insert_id;
    
    // Start background processing
    $cmd = "php ../cron/process-import.php " . $batch_id . " > /dev/null 2>&1 &";
    shell_exec($cmd);
    
    return $batch_id;
}

$db->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bulk Email Upload</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container">
            <a class="navbar-brand" href="#">Admin Panel</a>
            <div class="navbar-nav ms-auto">
                <a class="nav-item nav-link" href="dashboard.php">Dashboard</a>
                <a class="nav-item nav-link active" href="upload-emails.php">Upload Emails</a>
                <a class="nav-item nav-link" href="../logout.php">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <h2>Bulk Email Upload</h2>
        
        <?php echo $message; ?>
        
        <div class="row">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h4>Upload Email List</h4>
                    </div>
                    <div class="card-body">
                        <form method="POST" enctype="multipart/form-data">
                            <div class="mb-3">
                                <label>Batch Name</label>
                                <input type="text" name="batch_name" class="form-control" required 
                                       placeholder="e.g., Facebook Emails Batch 1">
                            </div>
                            
                            <div class="mb-3">
                                <label>Select CSV File</label>
                                <input type="file" name="email_file" class="form-control" accept=".csv,.txt" required>
                                <small class="text-muted">
                                    CSV format: email,password<br>
                                    Example: user1@gmail.com,password123<br>
                                    Max file size: 10MB
                                </small>
                            </div>
                            
                            <div class="mb-3">
                                <label>Or Paste Emails (One per line)</label>
                                <textarea name="email_text" class="form-control" rows="10" 
                                          placeholder="email:password&#10;user1@gmail.com:pass123&#10;user2@yahoo.com:pass456"></textarea>
                                <small class="text-muted">Format: email:password (one per line)</small>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">Upload & Process</button>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h4>Recent Imports</h4>
                    </div>
                    <div class="card-body">
                        <div class="list-group">
                            <?php while($import = $recent_imports->fetch_assoc()): ?>
                            <div class="list-group-item">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1"><?php echo $import['batch_name']; ?></h6>
                                    <small><?php echo date('M d, H:i', strtotime($import['import_date'])); ?></small>
                                </div>
                                <p class="mb-1">
                                    File: <?php echo $import['filename']; ?><br>
                                    Status: 
                                    <span class="badge bg-<?php 
                                        echo $import['status'] == 'completed' ? 'success' : 
                                              ($import['status'] == 'processing' ? 'warning' : 'danger'); ?>">
                                        <?php echo ucfirst($import['status']); ?>
                                    </span>
                                </p>
                                <small>By: <?php echo $import['importer_name']; ?></small>
                            </div>
                            <?php endwhile; ?>
                        </div>
                    </div>
                </div>
                
                <div class="card mt-3">
                    <div class="card-header">
                        <h4>Quick Stats</h4>
                    </div>
                    <div class="card-body">
                        <?php
                        $db = getDB();
                        $stats = $db->query("
                            SELECT 
                                COUNT(*) as total_emails,
                                SUM(CASE WHEN status = 'available' THEN 1 ELSE 0 END) as available,
                                SUM(CASE WHEN status = 'used' THEN 1 ELSE 0 END) as used,
                                SUM(CASE WHEN status = 'assigned' THEN 1 ELSE 0 END) as assigned
                            FROM email_pool
                        ")->fetch_assoc();
                        ?>
                        <p><strong>Total Emails:</strong> <?php echo $stats['total_emails']; ?></p>
                        <p><strong>Available:</strong> <?php echo $stats['available']; ?></p>
                        <p><strong>Assigned:</strong> <?php echo $stats['assigned']; ?></p>
                        <p><strong>Used:</strong> <?php echo $stats['used']; ?></p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>