<?php
// Disable direct access
if (basename($_SERVER['PHP_SELF']) == basename(__FILE__)) {
    die('Direct access not allowed');
}

function createDatabaseTables() {
    require_once 'config/database.php';
    
    try {
        $db = getDB();
        
        // Users table
        $sql = "CREATE TABLE IF NOT EXISTS users (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(100) NOT NULL,
            email VARCHAR(100) UNIQUE NOT NULL,
            phone VARCHAR(20) UNIQUE NOT NULL,
            country_code VARCHAR(10) NOT NULL,
            country_name VARCHAR(50) NOT NULL,
            password VARCHAR(255) NOT NULL,
            telegram_id VARCHAR(100) UNIQUE,
            telegram_username VARCHAR(100),
            email_verified BOOLEAN DEFAULT 0,
            verification_token VARCHAR(100),
            verification_expiry DATETIME,
            notification_preference ENUM('email', 'telegram', 'both') DEFAULT 'both',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
            INDEX idx_email (email),
            INDEX idx_phone (phone),
            INDEX idx_telegram (telegram_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ Users table created<br>";
        }
        
        // Email pool table
        $sql = "CREATE TABLE IF NOT EXISTS email_pool (
            id INT PRIMARY KEY AUTO_INCREMENT,
            email_address VARCHAR(100) UNIQUE NOT NULL,
            email_password VARCHAR(255) NOT NULL,
            status ENUM('available', 'assigned', 'used', 'blocked') DEFAULT 'available',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            last_used DATETIME NULL,
            INDEX idx_status (status),
            INDEX idx_email (email_address)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ Email pool table created<br>";
        }
        
        // Services table
        $sql = "CREATE TABLE IF NOT EXISTS services (
            id INT PRIMARY KEY AUTO_INCREMENT,
            service_name VARCHAR(100) UNIQUE NOT NULL,
            description TEXT,
            price_per_email DECIMAL(10,2),
            telegram_command VARCHAR(50),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_service_name (service_name)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ Services table created<br>";
            
            // Insert default services
            $services = [
                ['Facebook Verification', 'Email accounts for Facebook verification', 0.50, 'facebook'],
                ['Instagram Verification', 'Email accounts for Instagram verification', 0.45, 'instagram'],
                ['LinkedIn Verification', 'Email accounts for LinkedIn verification', 0.60, 'linkedin'],
                ['X (Twitter) Verification', 'Email accounts for X/Twitter verification', 0.55, 'twitter'],
                ['Snapchat Opening', 'Email accounts for Snapchat account creation', 0.40, 'snapchat'],
                ['TikTok Opening', 'Email accounts for TikTok account creation', 0.35, 'tiktok']
            ];
            
            foreach ($services as $service) {
                $stmt = $db->prepare("INSERT IGNORE INTO services (service_name, description, price_per_email, telegram_command) VALUES (?, ?, ?, ?)");
                $stmt->bind_param("ssds", $service[0], $service[1], $service[2], $service[3]);
                $stmt->execute();
            }
            echo "✓ Default services inserted<br>";
        }
        
        // User purchases table
        $sql = "CREATE TABLE IF NOT EXISTS user_purchases (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT,
            service_id INT,
            email_pool_id INT,
            quantity INT NOT NULL,
            used_count INT DEFAULT 0,
            purchase_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            expiry_date DATETIME,
            status ENUM('active', 'expired', 'used') DEFAULT 'active',
            telegram_message_id VARCHAR(100),
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
            FOREIGN KEY (email_pool_id) REFERENCES email_pool(id) ON DELETE SET NULL,
            INDEX idx_user_service (user_id, service_id),
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ User purchases table created<br>";
        }
        
        // Email usage log
        $sql = "CREATE TABLE IF NOT EXISTS email_usage_log (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT,
            email_pool_id INT,
            service_id INT,
            action_type VARCHAR(50),
            details TEXT,
            ip_address VARCHAR(45),
            telegram_message_id VARCHAR(100),
            used_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
            FOREIGN KEY (email_pool_id) REFERENCES email_pool(id) ON DELETE SET NULL,
            FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE SET NULL,
            INDEX idx_user_date (user_id, used_at),
            INDEX idx_email (email_pool_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ Email usage log table created<br>";
        }
        
        // Telegram sessions
        $sql = "CREATE TABLE IF NOT EXISTS telegram_sessions (
            id INT PRIMARY KEY AUTO_INCREMENT,
            telegram_id VARCHAR(100) UNIQUE NOT NULL,
            chat_id BIGINT,
            step VARCHAR(50),
            temp_data TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_telegram_id (telegram_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ Telegram sessions table created<br>";
        }
        
        // Telegram message logs
        $sql = "CREATE TABLE IF NOT EXISTS telegram_logs (
            id INT PRIMARY KEY AUTO_INCREMENT,
            telegram_id VARCHAR(100),
            chat_id BIGINT,
            message_type VARCHAR(50),
            message TEXT,
            direction ENUM('incoming', 'outgoing'),
            sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_telegram_chat (telegram_id, chat_id),
            INDEX idx_sent_at (sent_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ Telegram logs table created<br>";
        }
        
        // File uploads
        $sql = "CREATE TABLE IF NOT EXISTS file_uploads (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT,
            filename VARCHAR(255),
            original_name VARCHAR(255),
            file_path VARCHAR(500),
            file_size INT,
            file_type VARCHAR(50),
            upload_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            status ENUM('pending', 'processing', 'completed', 'failed') DEFAULT 'pending',
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_status (user_id, status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if ($db->query($sql)) {
            echo "✓ File uploads table created<br>";
        }
        
        echo "<br><strong>All tables created successfully!</strong>";
        
        $db->close();
        
    } catch (Exception $e) {
        echo "Error: " . $e->getMessage();
    }
}

// If accessed directly
if (isset($_GET['action']) && $_GET['action'] == 'create') {
    createDatabaseTables();
}
?>