<?php
require_once 'config/database.php';

$countries = [
    'BD' => 'Bangladesh (+880)',
    'US' => 'United States (+1)',
    'UK' => 'United Kingdom (+44)',
    'IN' => 'India (+91)',
    'PK' => 'Pakistan (+92)',
    // Add more countries as needed
];

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $name = sanitize($_POST['name']);
    $email = sanitize($_POST['email']);
    $phone = sanitize($_POST['phone']);
    $country_code = sanitize($_POST['country_code']);
    $country_name = sanitize($_POST['country_name']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    
    // Validation
    if (empty($name) || empty($email) || empty($phone) || empty($password)) {
        $error = "All fields are required!";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "Invalid email format!";
    } elseif ($password !== $confirm_password) {
        $error = "Passwords do not match!";
    } elseif (strlen($password) < 8) {
        $error = "Password must be at least 8 characters!";
    } else {
        $db = getDB();
        
        // Check if email already exists
        $stmt = $db->prepare("SELECT id FROM users WHERE email = ? OR phone = ?");
        $stmt->bind_param("ss", $email, $phone);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error = "Email or Phone already registered!";
        } else {
            // Hash password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $verification_token = bin2hex(random_bytes(50));
            $verification_expiry = date('Y-m-d H:i:s', strtotime('+24 hours'));
            
            // Insert user
            $stmt = $db->prepare("INSERT INTO users (name, email, phone, country_code, country_name, password, verification_token, verification_expiry) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("ssssssss", $name, $email, $phone, $country_code, $country_name, $hashed_password, $verification_token, $verification_expiry);
            
            if ($stmt->execute()) {
                // Send verification email
                $verification_link = SITE_URL . "verify.php?token=" . $verification_token;
                $subject = "Email Verification - Email Service";
                $message = "Hello $name,<br><br>";
                $message .= "Please verify your email by clicking the link below:<br>";
                $message .= "<a href='$verification_link'>$verification_link</a><br><br>";
                $message .= "This link will expire in 24 hours.<br><br>";
                $message .= "Thank you!";
                
                // For now, we'll just display the link
                $success = "Registration successful! Verification link: " . $verification_link;
                
                // In production, use PHPMailer or similar
                // mail($email, $subject, $message, "From: " . FROM_EMAIL);
                
            } else {
                $error = "Registration failed. Please try again.";
            }
        }
        $db->close();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Email Service</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background: #f5f5f5; }
        .register-container { max-width: 500px; margin: 50px auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
    </style>
</head>
<body>
    <div class="container">
        <div class="register-container">
            <h2 class="text-center mb-4">Create Account</h2>
            
            <?php if ($error): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
            <?php endif; ?>
            
            <form method="POST" action="">
                <div class="mb-3">
                    <label>Full Name</label>
                    <input type="text" name="name" class="form-control" required>
                </div>
                
                <div class="mb-3">
                    <label>Email Address</label>
                    <input type="email" name="email" class="form-control" required>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-4">
                        <label>Country Code</label>
                        <select name="country_code" class="form-control" id="country_code" onchange="updateCountry()" required>
                            <option value="">Select</option>
                            <option value="+880">+880 (BD)</option>
                            <option value="+1">+1 (US)</option>
                            <option value="+44">+44 (UK)</option>
                            <option value="+91">+91 (IN)</option>
                            <option value="+92">+92 (PK)</option>
                        </select>
                    </div>
                    <div class="col-md-8">
                        <label>Phone Number</label>
                        <input type="text" name="phone" class="form-control" required>
                    </div>
                </div>
                
                <div class="mb-3">
                    <label>Country</label>
                    <input type="text" name="country_name" id="country_name" class="form-control" readonly required>
                </div>
                
                <div class="mb-3">
                    <label>Password</label>
                    <input type="password" name="password" class="form-control" required minlength="8">
                </div>
                
                <div class="mb-3">
                    <label>Confirm Password</label>
                    <input type="password" name="confirm_password" class="form-control" required>
                </div>
                
                <button type="submit" class="btn btn-primary w-100">Register</button>
            </form>
            
            <div class="text-center mt-3">
                <p>Already have an account? <a href="login.php">Login here</a></p>
            </div>
        </div>
    </div>

    <script>
        const countryMap = {
            '+880': 'Bangladesh',
            '+1': 'United States',
            '+44': 'United Kingdom',
            '+91': 'India',
            '+92': 'Pakistan'
        };
        
        function updateCountry() {
            const code = document.getElementById('country_code').value;
            document.getElementById('country_name').value = countryMap[code] || '';
        }
    </script>
</body>
</html>